#!/bin/bash
#
# Arguments: <on|off|status>
# Returns:   1 for ON, 2 for off, 0 for tests and -1/255 on errors
#

# settings
PING_INTERVAL=1
SWITCH_COUNT=60
ACK_COUNT=3

GPIO_PIN=18
cmd=
host=
state=

CMD_ON=on
CMD_OFF=off
CMD_STAT=status

ARGS=2
TOOL_DIR=/home/pi
SWITCH_LOCK=/tmp/.switch.lock

ERROR=-1
ON=1
OFF=2

# put output to log file for debugging purposes
set -x
exec &> /tmp/switch.log

# loggs error and exits
# param1 is the message
# param2 is a flag for lock file deletion
error()
{
    logger "error: $1"
    if [ -z $2 ]; then
        rm -f $SWITCH_LOCK
    fi
    exit $ERROR
}

# cleans up on exit
cleanup()
{
    rm -f $SWITCH_LOCK
}

# setup GPIO interface
gpio_init()
{
    # set direction, pull etc.
    echo $GPIO_PIN > /sys/class/gpio/export
    sleep 1
    echo out > /sys/class/gpio/gpio$GPIO_PIN/direction
    echo 1 > /sys/class/gpio/gpio$GPIO_PIN/active_low
}

# simulates a push button action via GPIO
gpio_push()
{
    # set signal
    echo 1 > /sys/class/gpio/gpio$GPIO_PIN/value
    # wait a second
    sleep 1
    # set signal back
    echo 0 > /sys/class/gpio/gpio$GPIO_PIN/value
}

# exits with actual state
# param1 is a exit flag
status()
{
    ping -i $PING_INTERVAL -c $ACK_COUNT $host &>/dev/null
    local res=$?
    case $res in
        0)
            state=$ON
            ;;
        *)
            state=$OFF
            ;;
    esac
    if ! [ -z $1 ]; then
        cleanup
        exit $state
    fi
}

# turns the machine on
turn_on()
{
    status
    if [ "$state" == "$OFF" ]; then
        gpio_init
        gpio_push
        local counter=$(($SWITCH_COUNT/$ACK_COUNT))
        while [ $counter -gt 0 ]; do
            status
            if [ "$state" == "$ON" ]; then
                cleanup
                exit $state
            fi
            let "counter-=1"
        done
    fi
    cleanup
    exit $state
}

# turns the machine on
turn_off()
{
    status
    if [ "$state" == "$ON" ]; then
        gpio_init
        gpio_push
        local counter=$(($SWITCH_COUNT/$ACK_COUNT))
        while [ $counter -gt 0 ]; do
            status
            if [ "$state" == "$OFF" ]; then
                cleanup
                exit $state
            fi
            let "counter-=1"
        done
    fi
    cleanup
    exit $state
}

# handles signals
sig_handler()
{
    cleanup
    exit
}

# test arguments
if [ $# -ne $ARGS ]; then
    error "invalid argument count $#" 0
fi

# test on parallel execution
if [ -e $SWITCH_LOCK ]; then
    error "lockfile exists" 0
else
    touch $SWITCH_LOCK
fi

# trap for signals
trap 'sig_handler' SIGINT EXIT

# execution
cmd=$1
host=$2
case $cmd in
    $CMD_ON)
        turn_on
        ;;
    $CMD_OFF)
        turn_off
        ;;
    $CMD_STAT)
        status 1
        ;;
    *)
        error "command $cmd not recognised"
        ;;
esac
