#!/bin/bash

INSTALL_DIR=/usr/share/shgen
LOCAL_DIR=~/.shgen
SETTINGS=$LOCAL_DIR/settings

ERR_FILE=/tmp/shgen.tmp

E_NOARGS=1
E_CREATE_DIR=2
E_GENERATE=3

LANG=C

create_dir()
{
	if ! [ -e $LOCAL_DIR ]; then
		# create local dir
		mkdir $LOCAL_DIR 2>$ERR_FILE 1>/dev/null ||\
		 error "creating local dir failed" $E_CREATE_DIR $ERR_FILE
		# copy files and set permisson
		cp $INSTALL_DIR/skeleton $INSTALL_DIR/custom $INSTALL_DIR/GPLv*\
		 $INSTALL_DIR/lib $LOCAL_DIR/ 2>$ERR_FILE 1>/dev/null ||\
		 error "copying files failed" $E_CREATE_DIR $ERR_FILE
	fi
}

error()
{
	echo -n "Error: $1" 1>&2
	[ -z $3 ] && echo "" || echo -e ":\n$(cat $3)" 1>&2
	exit $2
}

generate()
{
	# read/source settings file
	. $SETTINGS

	# generate main script according to the settings
	cp $LOCAL_DIR/skeleton $script 2>$ERR_FILE 1>/dev/null ||\
	 error "creating script failed" $E_GENERATE $ERR_FILE
	chmod +x $script 2>$ERR_FILE 1>/dev/null ||\
	 error "setting script permisson failed" $E_GENERATE $ERR_FILE

	###################################################
	# lib
	###################################################
	if [ $lib -eq 1 ]; then
		# generate lib script according to the settings
		cp $LOCAL_DIR/lib $lfile 2>$ERR_FILE 1>/dev/null ||\
		 error "creating lib failed" $E_GENERATE $ERR_FILE
		chmod +x $lfile 2>$ERR_FILE 1>/dev/null ||\
		 error "setting lib permisson failed" $E_GENERATE $ERR_FILE
		# write settings
		sed "s:%lib%:. $lfile:" -i $script 2>$ERR_FILE\
		 1>/dev/null || error "writing lib failed" $E_GENERATE $ERR_FILE
	else
		# delete placeholder
		sed "/%lib%/d" -i $script 2>$ERR_FILE 1>/dev/null || error\
		 "deleting lib placeholder failed" $E_GENERATE $ERR_FILE
	fi

	###################################################
	# shabang
	###################################################
	sed "s:%shabang%:#!$shabang:" -i $script 2>$ERR_FILE 1>/dev/null ||\
	 error "writing shabang failed" $E_GENERATE $ERR_FILE
	if [ $lib -eq 1 ]; then
		sed "s:%shabang%:#!$shabang:" -i $lfile 2>$ERR_FILE\
		 1>/dev/null || error "writing shabang in lib failed" $E_GENERATE\
		 $ERR_FILE
	fi

	###################################################
	# header
	###################################################
	if [ $header -eq 1 ]; then
		# description, copyright
		local copyright="# Copyright (C) $(date +%Y) $USER"
		sed -e "s/%description%/# $description/" -e\
		 "s/%copyright%/$copyright/" -i $script 2>$ERR_FILE 1>/dev/null\
		 || error "writing header failed" $E_GENERATE $ERR_FILE
		# same to lib
		if [ $lib -eq 1 ]; then
			sed "s/%copyright%/$copyright/" -i $lfile 2>$ERR_FILE\
			 1>/dev/null || error "writing header in lib failed"\
			 $E_GENERATE $ERR_FILE
		fi

		# license
		case $license in
			GPLv2|GPLv3|custom)
				sed "/%license%/r $LOCAL_DIR/$license" -i $script\
				 2>$ERR_FILE 1>/dev/null || error "writing header failed"\
				 $E_GENERATE $ERR_FILE
				# same to lib
				if [ $lib -eq 1 ]; then
					sed "/%license%/r $LOCAL_DIR/$license" -i $lfile\
					 2>$ERR_FILE 1>/dev/null || error\
					 "writing header in lib failed" $E_GENERATE $ERR_FILE
				fi
				;;
		esac
		sed "/%license%/d" -i $script 2>$ERR_FILE 1>/dev/null || error\
		 "deleting header placeholder failed" $E_GENERATE $ERR_FILE
		# same to lib
		if [ $lib -eq 1 ]; then
			sed "/%license%/d" -i $lfile 2>$ERR_FILE 1>/dev/null || error\
			 "deleting header placeholder in lib failed" $E_GENERATE\
			 $ERR_FILE
		fi
	else
		# delete placeholder
		sed -e "/%description%/d" -e "/%copyright%/d" -e "/%license%/d"\
		 -i $script 2>$ERR_FILE 1>/dev/null || error\
		 "deleting header placeholder failed" $E_GENERATE $ERR_FILE
		# same to lib
		if [ $lib -eq 1 ]; then
			sed -e "/%copyright%/d" -e "/%license%/d" -i $lfile 2>$ERR_FILE\
			 1>/dev/null || error "deleting header placeholder in lib failed"\
			 $E_GENERATE $ERR_FILE
		fi
	fi

	# create temp file
	local tmpf=$(mktemp 2>$ERR_FILE)
	[ -e $tmpf ] || error "creating tmpfile failed" $E_GENERATE $ERR_FILE

	###################################################
	# debug
	###################################################
	if [ $debug -eq 1 ]; then
		cat << EOF >/$tmpf
exec 3>$dfile
BASH_XTRACEFD=3
set -x
EOF
		if [ $lib -eq 1 ]; then
			sed -e "/%debug%/r $tmpf" -e "/%debug%/d" -i $lfile\
			 2>$ERR_FILE 1>/dev/null || error\
			 "writing debug in lib failed" $E_GENERATE $ERR_FILE
			# delete placeholder in script
			sed "/%debug%/d" -i $script 2>$ERR_FILE 1>/dev/null ||\
			 error "deleting debug placeholder failed" $E_GENERATE\
			 $ERR_FILE
		else
			sed -e "/%debug%/r $tmpf" -e "/%debug%/d" -i $script\
			 2>$ERR_FILE 1>/dev/null || error "writing debug failed"\
			 $E_GENERATE $ERR_FILE
		fi
	else
		# delete placeholder
		sed "/%debug%/d" -i $script 2>$ERR_FILE 1>/dev/null ||\
		 error "deleting debug placeholder failed" $E_GENERATE $ERR_FILE
	fi

	###################################################
	# usage()
	###################################################
	if [ "$common1" == "usage()" ]; then
		cat << _EOF >/$tmpf
usage()
{
	cat << EOF
Usage: \$0 [-h]

Options:  -h  show help (resp. this message)
EOF
}

_EOF
		# place description if available
		if [ $header -eq 1 ]; then
			sed -e "4i$description" -i $tmpf
		fi
		sed -e "/%common1%/r $tmpf" -e "/%common1%/d" -i $script 2>$ERR_FILE\
		 1>/dev/null || error "writing common1 in lib failed" $E_GENERATE\
		 $ERR_FILE
	else
		# delete placeholder
		if [ $lib -eq 1 ]; then
			sed "/%common1%/d" -i $lfile 2>$ERR_FILE 1>/dev/null || error\
			 "deleting common1 placeholder in lib failed" $E_GENERATE $ERR_FILE
		fi
		sed "/%common1%/d" -i $script 2>$ERR_FILE 1>/dev/null || error\
		 "deleting common1 placeholder failed" $E_GENERATE $ERR_FILE
	fi

	###################################################
	# log()
	###################################################
	if [ "$common2" == "log()" ]; then
		cat << _EOF >/$tmpf
log()
{
	echo -n "\$(basename \$0): LOG: \$1"
	[ -z \$2 ] || { echo; echo "\$2"; } && { echo; }
}

_EOF
		if [ $lib -eq 1 ]; then
			sed -e "/%common2%/r $tmpf" -e "/%common2%/d" -i $lfile\
			 2>$ERR_FILE 1>/dev/null || error "writing common2 failed"\
			 $E_GENERATE $ERR_FILE
			# delete placeholder in script
			sed "/%common2%/d" -i $script 2>$ERR_FILE 1>/dev/null ||\
			 error "deleting common2 placeholder failed" $E_GENERATE\
			 $ERR_FILE
		else
			sed -e "/%common2%/r $tmpf" -e "/%common2%/d" -i $script\
			 2>$ERR_FILE 1>/dev/null || error "writing common2 in lib failed"\
			 $E_GENERATE $ERR_FILE
		fi
	else
		# delete placeholder
		if [ $lib -eq 1 ]; then
			sed "/%common2%/d" -i $lfile 2>$ERR_FILE 1>/dev/null || error\
			 "deleting common2 placeholder in lib failed" $E_GENERATE $ERR_FILE
		fi
		sed "/%common2%/d" -i $script 2>$ERR_FILE 1>/dev/null || error\
		 "deleting common2 placeholder failed" $E_GENERATE $ERR_FILE
	fi

	###################################################
	# error()
	###################################################
	if [ "$common3" == "error()" ]; then
		cat << _EOF >/$tmpf
error()
{
	echo -n "\$(basename \$0): ERROR: \$1"
	[ -z \$2 ] || { echo; echo "\$2"; } && { echo; }
	exit 1
}

_EOF
		if [ $lib -eq 1 ]; then
			sed -e "/%common3%/r $tmpf" -e "/%common3%/d" -i $lfile\
			 2>$ERR_FILE 1>/dev/null || error "writing common3 failed"\
			 $E_GENERATE $ERR_FILE
			# delete placeholder in script
			sed "/%common3%/d" -i $script 2>$ERR_FILE 1>/dev/null ||\
			 error "deleting common3 placeholder failed" $E_GENERATE\
			 $ERR_FILE
		else
			sed -e "/%common3%/r $tmpf" -e "/%common3%/d" -i $script\
			 2>$ERR_FILE 1>/dev/null || error "writing common3 in lib failed"\
			 $E_GENERATE $ERR_FILE
		fi
	else
		# delete placeholder
		if [ $lib -eq 1 ]; then
			sed "/%common3%/d" -i $lfile 2>$ERR_FILE 1>/dev/null || error\
			 "deleting common3 placeholder in lib failed" $E_GENERATE $ERR_FILE
		fi
		sed "/%common3%/d" -i $script 2>$ERR_FILE 1>/dev/null || error\
		 "deleting common3 placeholder failed" $E_GENERATE $ERR_FILE
	fi

	###################################################
	# warning()
	###################################################
	if [ "$common4" == "warning()" ]; then
		cat << _EOF >/$tmpf
warning()
{
	echo -n "\$(basename \$0): WARNING: \$1"
	[ -z \$2 ] || { echo; echo "\$2"; } && { echo; }
}

_EOF
		if [ $lib -eq 1 ]; then
			sed -e "/%common4%/r $tmpf" -e "/%common4%/d" -i $lfile\
			 2>$ERR_FILE 1>/dev/null || error "writing common4 failed"\
			 $E_GENERATE $ERR_FILE
			# delete placeholder in script
			sed "/%common4%/d" -i $script 2>$ERR_FILE 1>/dev/null ||\
			 error "deleting common4 placeholder failed" $E_GENERATE\
			 $ERR_FILE
		else
			sed -e "/%common4%/r $tmpf" -e "/%common4%/d" -i $script\
			 2>$ERR_FILE 1>/dev/null || error "writing common4 in lib failed"\
			 $E_GENERATE $ERR_FILE
		fi
	else
		# delete placeholder
		if [ $lib -eq 1 ]; then
			sed "/%common4%/d" -i $lfile 2>$ERR_FILE 1>/dev/null || error\
			 "deleting common4 placeholder in lib failed" $E_GENERATE $ERR_FILE
		fi
		sed "/%common4%/d" -i $script 2>$ERR_FILE 1>/dev/null || error\
		 "deleting common4 placeholder failed" $E_GENERATE $ERR_FILE
	fi

	# delete temp file
	rm $tmpf &>/dev/null

	###################################################
	# option processing
	###################################################
	if [ "$common1" == "usage()" ]; then
		cat << _EOF >/$tmpf
while getopts ":h" opt; do
	case \$opt in
		h)
			usage
			;;
		*)
			usage
			;;
	esac
done
_EOF
		sed -e "/%options%/r $tmpf" -e "/%options%/d" -i $script 2>$ERR_FILE\
		 1>/dev/null || error "writing options failed" $E_GENERATE $ERR_FILE
	else
		# delete placeholder
		sed "/%options%/d" -i $script 2>$ERR_FILE 1>/dev/null || error\
		 "deleting options placeholder failed" $E_GENERATE $ERR_FILE
	fi

	# delete temp file
	rm $tmpf &>/dev/null
}

usage()
{
	cat <<EOF
Usage: $0 -<c|g>

Options:  -c  create the local directory ~/.shgen
          -g  generate shell scripts according to the settings
EOF
	exit $E_NOARGS
}

[ $# -eq 0 ] && usage

while getopts ":cg" opt; do
	case $opt in
		c)
			create_dir
			;;
		g)
			generate
			;;
		*)
			usage
			;;
	esac
done

exit 0
